import { useState, useEffect, useRef } from 'react'
import { useNavigate } from 'react-router-dom'
import { Scissors, User, Calendar, Image as ImageIcon, Package, CheckCircle, Award, Clock, Users } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyCourses.css'

const MyCourses = () => {
  const navigate = useNavigate()
  const [courses, setCourses] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('sewingCourses')
    const defaultCourses = [
      {
        id: 1,
        project: 'Платье-футляр',
        teacher: 'Елена Иванова',
        completionDate: '2025-05-15',
        status: 'active',
        resultPhoto: null
      },
      {
        id: 2,
        project: 'Блузка с воротником',
        teacher: 'Мария Петрова',
        completionDate: '2025-04-20',
        status: 'completed',
        resultPhoto: 'https://images.unsplash.com/photo-1594633313593-bab3825d0caf?w=400'
      },
      {
        id: 3,
        project: 'Юбка-карандаш',
        teacher: 'Анна Соколова',
        completionDate: '2025-03-10',
        status: 'completed',
        resultPhoto: 'https://images.unsplash.com/photo-1594633313593-bab3825d0caf?w=400'
      }
    ]
    
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        const validCourses = parsed.filter(course => 
          course.project && course.teacher && course.completionDate &&
          course.completionDate.match(/^\d{4}-\d{2}-\d{2}$/)
        )
        if (validCourses.length > 0) {
          setCourses(parsed)
        } else {
          setCourses(defaultCourses)
          localStorage.setItem('sewingCourses', JSON.stringify(defaultCourses))
        }
      } catch (e) {
        setCourses(defaultCourses)
        localStorage.setItem('sewingCourses', JSON.stringify(defaultCourses))
      }
    } else {
      setCourses(defaultCourses)
      localStorage.setItem('sewingCourses', JSON.stringify(defaultCourses))
    }
  }, [])

  const formatDate = (dateString) => {
    if (!dateString) return 'Не указано'
    try {
      const [year, month, day] = dateString.split('-').map(Number)
      if (!year || !month || !day) return 'Не указано'
      
      const date = new Date(year, month - 1, day)
      if (isNaN(date.getTime())) return 'Не указано'
      
      return date.toLocaleDateString('ru-RU', { 
        day: 'numeric', 
        month: 'long', 
        year: 'numeric' 
      })
    } catch (e) {
      return 'Не указано'
    }
  }

  return (
    <div className="my-courses-page-sewing">
      <div className="container">
        <section className="page-header-sewing">
          <div className="header-badge-sewing">
            <Scissors size={20} />
            <span>ВАШИ ПРОЕКТЫ</span>
          </div>
          <h1 className="page-title-sewing">
            Мои <span className="title-accent-sewing">курсы</span>
          </h1>
          <p className="page-description-sewing">
            Отслеживайте прогресс ваших проектов и делитесь результатами
          </p>
        </section>

        <ImageSlider />

        <MasterClassSection />

        {courses.length === 0 ? (
          <div className="empty-state-sewing">
            <div className="empty-icon-wrapper-sewing">
              <Package size={64} />
            </div>
            <h2 className="empty-title-sewing">Нет курсов</h2>
            <p className="empty-text-sewing">Запишитесь на первый курс по шитью</p>
            <button 
              className="empty-button-sewing" 
              onClick={() => navigate('/enroll')}
            >
              Записаться на курс
            </button>
          </div>
        ) : (
          <section className="courses-list-sewing">
            <div className="courses-section-header-sewing">
              <h2 className="courses-section-title-sewing">Мои записи на курсы</h2>
              <p className="courses-section-description-sewing">
                Отслеживайте прогресс ваших проектов и делитесь результатами
              </p>
            </div>
            <div className="courses-grid-sewing">
              {courses.map((course) => (
                <CourseCard
                  key={course.id}
                  course={course}
                  formatDate={formatDate}
                />
              ))}
            </div>
          </section>
        )}
      </div>
    </div>
  )
}

const CourseCard = ({ course, formatDate }) => {
  return (
    <div className={`course-item-sewing ${course.status}`}>
      <div className="course-header-sewing">
        <div className="course-icon-wrapper-sewing">
          <Scissors size={32} />
        </div>
        <div className="course-info-sewing">
          <div className="course-number-sewing">#{course.id}</div>
          <div className={`status-badge-sewing ${course.status}`}>
            {course.status === 'active' ? (
              <>
                <CheckCircle size={16} />
                <span>В процессе</span>
              </>
            ) : (
              <>
                <Award size={16} />
                <span>Завершен</span>
              </>
            )}
          </div>
        </div>
      </div>
      
      <h3 className="course-title-sewing">{course.project}</h3>
      
      <div className="course-details-sewing">
        <div className="detail-item-sewing">
          <User size={18} />
          <div>
            <span className="detail-label-sewing">Преподаватель:</span>
            <span className="detail-value-sewing">{course.teacher}</span>
          </div>
        </div>
        <div className="detail-item-sewing">
          <Calendar size={18} />
          <div>
            <span className="detail-label-sewing">Дата завершения:</span>
            <span className="detail-value-sewing">{formatDate(course.completionDate)}</span>
          </div>
        </div>
      </div>
      
      {course.status === 'completed' && course.resultPhoto && (
        <div className="course-result-sewing">
          <div className="result-photo-wrapper-sewing">
            <img 
              src={course.resultPhoto} 
              alt={course.project}
              className="result-photo-sewing"
              onError={(e) => {
                e.target.style.display = 'none'
                e.target.nextSibling.style.display = 'flex'
              }}
            />
            <div className="result-photo-placeholder-sewing" style={{ display: 'none' }}>
              <ImageIcon size={48} />
            </div>
          </div>
          <div className="result-label-sewing">
            <ImageIcon size={16} />
            <span>Фото результата</span>
          </div>
        </div>
      )}
    </div>
  )
}

const MasterClassSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const masterclasses = [
    {
      title: 'Мастер-класс: Платье-футляр',
      instructor: 'Елена Иванова',
      duration: '4 часа',
      level: 'Средний',
      students: 24,
      image: 'linear-gradient(135deg, #E8B4B8 0%, #C4B5D9 100%)'
    },
    {
      title: 'Мастер-класс: Блузка с воротником',
      instructor: 'Мария Петрова',
      duration: '3 часа',
      level: 'Начальный',
      students: 18,
      image: 'linear-gradient(135deg, #C4B5D9 0%, #A8C4A2 100%)'
    },
    {
      title: 'Мастер-класс: Юбка-карандаш',
      instructor: 'Анна Соколова',
      duration: '5 часов',
      level: 'Продвинутый',
      students: 12,
      image: 'linear-gradient(135deg, #A8C4A2 0%, #E8B4B8 100%)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`masterclass-section-sewing ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="masterclass-header-sewing">
          <h2 className="masterclass-title-sewing">Ближайшие мастер-классы</h2>
          <p className="masterclass-description-sewing">
            Присоединяйтесь к интенсивным занятиям от лучших мастеров
          </p>
        </div>
        <div className="masterclass-grid-sewing">
          {masterclasses.map((masterclass, index) => (
            <div 
              key={index} 
              className="masterclass-card-sewing"
              style={{ 
                animationDelay: `${index * 0.15}s`,
                background: masterclass.image
              }}
            >
              <div className="masterclass-image-sewing"></div>
              <div className="masterclass-content-sewing">
                <div className="masterclass-level-sewing">{masterclass.level}</div>
                <h3 className="masterclass-card-title-sewing">{masterclass.title}</h3>
                <div className="masterclass-meta-sewing">
                  <div className="masterclass-meta-item-sewing">
                    <User size={16} />
                    <span>{masterclass.instructor}</span>
                  </div>
                  <div className="masterclass-meta-item-sewing">
                    <Clock size={16} />
                    <span>{masterclass.duration}</span>
                  </div>
                  <div className="masterclass-meta-item-sewing">
                    <Users size={16} />
                    <span>{masterclass.students} мест</span>
                  </div>
                </div>
                <button className="masterclass-button-sewing">
                  Записаться
                </button>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

export default MyCourses

